<?php

/**
 * This file is the main class file for AllinoneAccessibility
 *
 * @author Skynet Technologies USA LLC
 * @package allinoneaccessibility
 * 
 */
 
class AllinoneAccessibility
{
  /** @var modX|null A reference to the modX object */
  public $modx = null;
  
  /** @var array A collection of properties to adjust AllinoneAccessibility behaviour */
  public $config = array();
  
  /**
  * The AllinoneAccessibility Constructor.
  *
  * Creates a new AllinoneAccessibility object.
  *
  * @param modX &$modx A reference to the modX object.
  * @param array $config A collection of properties that modify AllinoneAccessibility behaviour.
  * @return AllinoneAccessibility A unique AllinoneAccessibility instance.
  */
  function __construct(modX &$modx,array $config = array())
  {
      $this->modx =& $modx;

      // allows you to set paths in different environments
      if ($modx->getOption('site_dev') == 1)
      {
          $basePath = $this->modx->getOption('allinoneaccessibility.core_path');
          $assetsUrl = $this->modx->getOption('allinoneaccessibility.assets_url');
        } else {
          $basePath = $this->modx->getOption('core_path').'components/allinoneaccessibility/';
          $assetsUrl = $this->modx->getOption('assets_url').'components/allinoneaccessibility/';
      }
            
      $this->config = array_merge(array(
          'basePath' => $basePath,
          'corePath' => $basePath,
          'modelPath' => $basePath.'model/',
          'processorsPath' => $basePath.'processors/',
          'templatesPath' => $basePath.'templates/',
          'chunksPath' => $basePath.'elements/chunks/',
          'jsUrl' => $assetsUrl.'js/',
          'cssUrl' => $assetsUrl.'css/',
          'assetsUrl' => $assetsUrl,
          'connectorUrl' => $assetsUrl.'connector.php',
      ),$config);

  }
  
  /**
  * Appends the processed Aioa Widget consent chunk to the generated resource output BODY tag
  */
  public function appendDisclaimer( $a = array() )
  {
    // setup default properties
    $a['class'] = $this->modx->getOption('class', $a, 'allinoneaccessibility');
    $a['allinoneName'] = "AllinOneaccessibility";
    
    $tpl = $this->modx->getOption('tpl', $a, 'allinoneAccessibility');

    $base_url = $this->modx->getOption('site_url');
    $siteUrl = preg_replace("/www\.|https?:\/\/|\/$|\/?\?.+|\/.+|^\./", '', $base_url);
    $hash = sha1($this->modx->getOption('base_url')."modx_accessibility_" . $siteUrl);
    
    /* get widget setting data form server using custom API- which is used in javascript */
    $api_url = 'https://skynettechnologies.com/add-ons/reports/setup/modx/api.php';
    $post_data = [
        'hash_val' => $hash,
    ];
    $options = array(
        'http' => array(
          'header'  => "Content-type: application/x-www-form-urlencoded\r\n",
          'method'  => 'POST',
          'content' => http_build_query($post_data),
        ),
      );
    $context  = stream_context_create($options);
    $result = file_get_contents($api_url, false, $context);
    $getData = json_decode($result, true);
    
    if($getData){
      $a['licenseKey'] = $getData['licensekey'];
      $a['colorCode'] = $getData['color'];
      $a['position'] = $getData['position'];
      $a['iconType'] = $getData['icon_type'];
      $a['iconSize'] = $getData['icon_size'];
    }else{
      $a['licenseKey'] = "";
      $a['colorCode'] = '#420083';
      $a['position'] = 'bottom_right';
      $a['iconType'] = 'aioa-icon-type-1';
      $a['iconSize'] = 'aioa-medium-icon';
    }
    
    /* get old chunk (script) replace with new setting value */
    if ($a['demo'] == 1) return $this->getChunk( $tpl, $a );
    
    $item = '</body>';
    $find[] = $item;
    $replace[] = $this->getChunk( $tpl, $a ) . "\n" . $item;

    $output = &$this->modx->resource->_output;
   
    $output = str_replace($find, $replace, $output);
  }
  
  /**
  * Processes a chunk. Attempts object first, then file based if not found
  *
  * @param string $name The name of the chunk
  * @param array $properties The settings for the chunk
  * @return string The content of the processed chunk
  */
  public function getChunk($name, $properties = array())
  {
      $chunk = null;
      
      if (!isset($this->chunks[$name]))
      {
          $chunk = $this->modx->getObject('modChunk', array('name' => $name));
          
          if (empty($chunk) || !is_object($chunk))
          {
            $chunk = $this->_getTplChunk($name);
            if ($chunk == false) return false;
          }
          
          $this->chunks[$name] = $chunk->getContent();
            
        } else {
          
          $o = $this->chunks[$name];
          $chunk = $this->modx->newObject('modChunk');
          $chunk->setContent($o);
      }
      
      $chunk->setCacheable(false);
      
      return $chunk->process($properties);
  }
  
  /**
  * Get the contents of a file based chunk
  *
  * @param string $name The name of the chunk
  * @param string $postfix The extension of the file based chunk
  * @return string The content of the file based chunk
  */
  private function _getTplChunk($name, $postfix = '.chunk.tpl')
  {
      $chunk = false;
      
      $f = $this->config['chunksPath'].strtolower($name).$postfix;
      
      if (file_exists($f))
      {
        $o = file_get_contents($f);
        $chunk = $this->modx->newObject('modChunk');
        $chunk->set('name',$name);
        $chunk->setContent($o);
      }
      
      return $chunk;
  }    
    
}