<?php

namespace AllinoneAccessibility\v3;

use MODX\Revolution\modX;

class AllinoneAccessibility
{
    /**
     * @var \modX $modx
     */
    public $modx;

    public $namespace = 'allinoneaccessibility';

    /**
     * @var array $config
     */
    public $config = [];
    /**
     * Constructs the AllinoneAccessibility object
     *
     * @param modX  &$modx  A reference to the modX object
     * @param array $config An array of configuration options
     */
    public function __construct(modX &$modx, array $config = [])
    {
        $this->modx =& $modx;

        $basePath = $this->modx->getOption(
            'allinoneaccessibility.core_path',
            $config,
            $this->modx->getOption('core_path') . 'components/allinoneaccessibility/'
        );
        $assetsUrl = $this->modx->getOption(
            'allinoneaccessibility.assets_url',
            $config,
            $this->modx->getOption('assets_url') . 'components/allinoneaccessibility/'
        );

        $this->config = array_merge(
            [
                'baseUrl' => $modx->getOption('base_url'),
                'basePath' => $basePath,
                'corePath' => $basePath,
                'srcPath' => $basePath . 'src/',
                'modelPath' => $basePath . 'src/Model/',
                'processorsPath' => $basePath . 'src/v3/Processors',
                'templatesPath' => $basePath . 'templates/',
                'chunksPath' => $basePath . 'elements/chunks/',
                'jsUrl' => $assetsUrl . 'js/',
                'cssUrl' => $assetsUrl . 'css/',
                'assetsUrl' => $assetsUrl,
                'managerUrl' => $this->modx->getOption('manager_url'),
                'media_source' => $this->modx->getOption(
                    'allinoneaccessibility.media_source',
                    null,
                    $this->modx->getOption('default_media_source'),
                    true
                )
            ],
            $config
        );

        $this->modx->addPackage('allinoneaccessibility', $this->getOption('modelPath'));
        $this->modx->lexicon->load('allinoneaccessibility:default');
        $this->autoload();
    }


    /**
     * Get a local configuration option or a namespaced system setting by key.
     *
     * @param string $key     The option key to search for.
     * @param array  $options An array of options that override local options.
     * @param mixed  $default The default value returned if the option is not found locally or as a
     *                        namespaced system setting; by default this value is null.
     *
     * @return mixed The option value or the default value specified.
     */
    public function getOption(string $key, $options = [], $default = null)
    {
        $option = $default;
        if (!empty($key)) {
            if ($options != null && array_key_exists($key, $options)) {
                $option = $options[$key];
            } elseif (array_key_exists($key, $this->config)) {
                $option = $this->config[$key];
            } elseif (array_key_exists("{$this->namespace}.{$key}", $this->modx->config)) {
                $option = $this->modx->getOption("{$this->namespace}.{$key}");
            }
        }
        return $option;
    }

    protected function autoload()
    {
        include_once $this->getOption('basePath') . 'vendor/autoload.php';
    }

    /**
     * Appends the processed Aioa Widget consent chunk to the generated resource output BODY tag
    */
  public function appendDisclaimer( $a = array() )
  {
    $a['class'] = $this->modx->getOption('class', $a, 'allinoneaccessibility');
    $a['allinoneName'] = "AllinOneaccessibility";
    
    $tpl = $this->modx->getOption('tpl', $a, 'allinoneAccessibility');

    $base_url = $this->modx->getOption('site_url');
    $siteUrl = preg_replace("/www\.|https?:\/\/|\/$|\/?\?.+|\/.+|^\./", '', $base_url);
    $hash = sha1($this->modx->getOption('base_url')."modx_accessibility_" . $siteUrl);
    
    $api_url = 'https://skynettechnologies.com/add-ons/reports/setup/modx/api.php';
    $post_data = [
        'hash_val' => $hash,
    ];
    $options = array(
        'http' => array(
          'header'  => "Content-type: application/x-www-form-urlencoded\r\n",
          'method'  => 'POST',
          'content' => http_build_query($post_data),
        ),
      );
    $context  = stream_context_create($options);
    $result = file_get_contents($api_url, false, $context);
    $getData = json_decode($result, true);
    
    if($getData){
      $a['licenseKey'] = $getData['licensekey'];
      $a['colorCode'] = $getData['color'];
      $a['position'] = $getData['position'];
      $a['iconType'] = $getData['icon_type'];
      $a['iconSize'] = $getData['icon_size'];
    }else{
      $a['licenseKey'] = "";
      $a['colorCode'] = '#420083';
      $a['position'] = 'bottom_right';
      $a['iconType'] = 'aioa-icon-type-1';
      $a['iconSize'] = 'aioa-medium-icon';
    }
   // print_r($getData);die;

    if ($a['demo'] == 1) return $this->getChunk( $tpl, $a );
    
    $item = '</body>';
    $find[] = $item;
    $replace[] = $this->getChunk( $tpl, $a ) . "\n" . $item;

    $output = &$this->modx->resource->_output;
   
    $output = str_replace($find, $replace, $output);
  }

  /**
  * Processes a chunk. Attempts object first, then file based if not found
  *
  * @param string $name The name of the chunk
  * @param array $properties The settings for the chunk
  * @return string The content of the processed chunk
  */
  public function getChunk($name, $properties = array())
  {
      $chunk = null;
      //print_r($this->chunks[$name]);die;
      if (!isset($this->chunks[$name]))
      {
       // print_r(1);die;
        $chunk = $this->modx->getObject('modChunk', array('name' => $name));
        
        if (empty($chunk) || !is_object($chunk))
        {
        $chunk = $this->_getTplChunk($name);
        if ($chunk == false) return false;
        }
        
        $this->chunks[$name] = $chunk->getContent();
            
    } else {
        
        $o = $this->chunks[$name];
        $chunk = $this->modx->newObject('modChunk');
        $chunk->setContent($o);
    }
      
      $chunk->setCacheable(false);
      
      return $chunk->process($properties);
  }

  /**
  * Get the contents of a file based chunk
  *
  * @param string $name The name of the chunk
  * @param string $postfix The extension of the file based chunk
  * @return string The content of the file based chunk
  */
  private function _getTplChunk($name, $postfix = '.chunk.tpl')
  {
      $chunk = false;
      //print_r($this->config['chunksPath']);die;
      $f = $this->config['chunksPath'].strtolower($name).$postfix;
      //print_r($f);die;
      if (file_exists($f))
      {
        
        $o = file_get_contents($f);
        $chunk = $this->modx->newObject('modChunk');
        $chunk->set('name',$name);
        $chunk->setContent($o);
      }
      
      return $chunk;
  }    
}
